-- This file contains the SQL statements necessary to create the 'control'
-- tables in the controldb schema for Dimension. It is intended to be run
-- by the web UI (or other) when it initially starts operation.
--
-- Copyright (c) 2014, 2015 WatchGuard Technologies, Inc.
-- All Rights Reserved
--
-- $Id //gsm/development/wsserver/mainline/src/schema/PostgreSQL/control.sql#5 $
--

CREATE SCHEMA controldb;
SET search_path TO controldb, userdb, public;

-- control schema configuration
-- maximum_revisions: number of device config revisions to keep
-- maximum_config_storage: maximum size of all config revisions per device in MB
CREATE TABLE g_control_config(
    maximum_revisions      INTEGER NOT NULL DEFAULT 10,
    maximum_config_storage INTEGER NOT NULL DEFAULT 5
);
INSERT INTO g_control_config (maximum_revisions, maximum_config_storage)
SELECT 10, 5 WHERE NOT EXISTS (SELECT * FROM g_control_config LIMIT 1);

-- contact personal details
CREATE SEQUENCE contacts_id_seq START WITH 1;
CREATE TABLE contacts (
   id               INTEGER NOT NULL DEFAULT NEXTVAL('contacts_id_seq'),
   name             TEXT NOT NULL,
   phone            TEXT,
   email            TEXT NOT NULL,
   fax              TEXT,
   email_alerts     BOOLEAN NOT NULL DEFAULT FALSE,
   device_id        INTEGER NOT NULL DEFAULT 0,
   PRIMARY KEY (id)
);
CREATE INDEX contacts_name_idx ON contacts(name);

-- managed devices inherits from public.g_deviceinfo
CREATE TABLE managed_devices (
    id               INTEGER NOT NULL,
    user_id          INTEGER NOT NULL REFERENCES users(id),
    active_features  TEXT DEFAULT NULL,
    ssh_pk           TEXT DEFAULT NULL,
    cluster_info     TEXT DEFAULT NULL,
    latitude         DOUBLE PRECISION DEFAULT NULL,
    longitude        DOUBLE PRECISION DEFAULT NULL,
    PRIMARY KEY (id)
) INHERITS (g_deviceinfo);
CREATE INDEX managed_devices_sn_idx ON managed_devices(sn);
CREATE INDEX managed_devices_cluster_idx ON managed_devices(cluster);

-- records device health reports
CREATE SEQUENCE device_health_id_seq START WITH 1;
CREATE TABLE device_health (
    id              INTEGER NOT NULL DEFAULT NEXTVAL('device_health_id_seq'),
    device_id       INTEGER NOT NULL,
    sample_time     TIMESTAMP WITHOUT TIME ZONE NOT NULL
                    DEFAULT date_trunc('second'::text, timezone('utc'::text, clock_timestamp())),
    role            INTEGER NOT NULL DEFAULT 0,
    uptime          INTEGER NOT NULL DEFAULT 0,
    load            INTEGER NOT NULL DEFAULT 0,
    cpu_util_1      INTEGER NOT NULL DEFAULT 0,
    cpu_util_5      INTEGER NOT NULL DEFAULT 0,
    cpu_util_15     INTEGER NOT NULL DEFAULT 0,
    mem_total       INTEGER NOT NULL DEFAULT 0,
    mem_used        INTEGER NOT NULL DEFAULT 0,
    mem_free        INTEGER NOT NULL DEFAULT 0
);
CREATE INDEX device_health_devsampletime_idx ON device_health (device_id, sample_time);

-- interface statistics
CREATE TABLE if_stats (
    health_id       INTEGER NOT NULL, -- FK won't work with partitions
    ifname          CHAR(8) NOT NULL,
    name            TEXT DEFAULT NULL,
    zone            TEXT DEFAULT NULL,
    recv_bytes      BIGINT NOT NULL DEFAULT 0,
    send_bytes      BIGINT NOT NULL DEFAULT 0,
    recv_packets    BIGINT NOT NULL DEFAULT 0,
    send_packets    BIGINT NOT NULL DEFAULT 0,
    error_drops     BIGINT NOT NULL DEFAULT 0,
    PRIMARY KEY (health_id, ifname)
);

-- rule to cleanup interface data when health record is removed
CREATE RULE cleanup_if_stats AS
    ON DELETE TO device_health
    DO DELETE FROM if_stats WHERE health_id = old.id;

-- vpn gateway statistics
CREATE TABLE vpn_gateway_stats (
    health_id               INTEGER NOT NULL, -- FK won't work with partitions
    name                    TEXT DEFAULT NULL,
    property                INTEGER NOT NULL,
    total_tunnel_num        INTEGER NOT NULL DEFAULT 0,
    active_tunnel_num       INTEGER NOT NULL DEFAULT 0,
    total_sent_bytes        BIGINT NOT NULL DEFAULT 0,
    total_received_bytes    BIGINT NOT NULL DEFAULT 0,
    PRIMARY KEY (health_id, name)
);

-- rule to clean up vpn gateway data when health record is removed
CREATE RULE cleanup_vpn_gateway_stats AS
    ON DELETE TO device_health
    DO DELETE FROM vpn_gateway_stats WHERE health_id = old.id;

-- table to act as a queue of requests for client XTM devices
CREATE SEQUENCE dxcp_request_queue_id_seq START WITH 1;
CREATE TABLE dxcp_request_queue (
    id              INTEGER NOT NULL DEFAULT NEXTVAL('dxcp_request_queue_id_seq'),
    device_id       INTEGER NOT NULL,
    instance_id     TEXT NOT NULL,
    user_id         TEXT NOT NULL, -- Login id of the user
    entry_time      TIMESTAMP WITHOUT TIME ZONE NOT NULL
                    DEFAULT date_trunc('second'::text, timezone('utc'::text, clock_timestamp())),
    sent_time       TIMESTAMP WITHOUT TIME ZONE DEFAULT NULL,
    request         TEXT NOT NULL,
    response        TEXT DEFAULT NULL,
    error_code      INTEGER DEFAULT NULL,
    PRIMARY KEY (id)
);
CREATE INDEX dxcp_request_queue_device_idx ON dxcp_request_queue (device_id);

-- client device configuration
CREATE SEQUENCE device_configs_config_id_seq START WITH 1;
CREATE TABLE device_configs (
    id              BIGINT PRIMARY KEY DEFAULT NEXTVAL('device_configs_config_id_seq'),
    device_id       INTEGER NOT NULL,
    device_version  TEXT NOT NULL,
    update_time     TIMESTAMP WITHOUT TIME ZONE NOT NULL
                    DEFAULT date_trunc('second'::text, timezone('utc'::text, clock_timestamp())),
    user_name       TEXT,
    user_comment    TEXT,
    config_revision INTEGER NOT NULL,
    config          TEXT NOT NULL,
    config_size     INTEGER NOT NULL,
    hash            TEXT NOT NULL DEFAULT '',
    annotation      TEXT DEFAULT NULL,
    preserved       BOOLEAN DEFAULT FALSE
);
CREATE INDEX device_configs_device_id_idx ON device_configs (device_id);

-- client interface configuration
CREATE TABLE if_configs (
    device_id       INTEGER NOT NULL,
    update_time     TIMESTAMP WITHOUT TIME ZONE NOT NULL
                    DEFAULT date_trunc('second'::text, timezone('utc'::text, clock_timestamp())),
    ifname          TEXT NOT NULL,
    name            TEXT NOT NULL,
    ip_assignment   TEXT NOT NULL,
    ip              TEXT NOT NULL DEFAULT '',
    netmask         TEXT NOT NULL DEFAULT '',
    PRIMARY KEY (device_id, ifname)
);

-- sequence for managed_vpns
CREATE SEQUENCE managed_vpn_seq START WITH 1;

-- sequence for vpn_gateways
CREATE SEQUENCE vpn_gateway_seq START WITH 1;

-- sequence for all other VPN associated tables
CREATE SEQUENCE wg_vpn_seq START WITH 1;

-- root table for managed hub-and spoke VPN
CREATE TABLE managed_vpns (
    id                    INTEGER NOT NULL DEFAULT NEXTVAL('managed_vpn_seq'),
    update_time           TIMESTAMP WITHOUT TIME ZONE NOT NULL
                          DEFAULT date_trunc('second'::text, timezone('utc'::text, clock_timestamp())),
    name                  TEXT NOT NULL,
    auth_method           INTEGER NOT NULL DEFAULT 1,
    psk                   TEXT NOT NULL,
    use_vif               BOOLEAN NOT NULL,
    PRIMARY KEY (id)
);

-- contains the custom phase1/phase2 security settings for a managed vpn
CREATE TABLE vpn_security_templates (
    id                    INTEGER NOT NULL DEFAULT NEXTVAL('wg_vpn_seq'),
    update_time           TIMESTAMP WITHOUT TIME ZONE NOT NULL
                          DEFAULT date_trunc('second'::text, timezone('utc'::text, clock_timestamp())),
    managed_vpn_id        INTEGER DEFAULT NULL REFERENCES managed_vpns(id) ON DELETE CASCADE,
    custom                BOOLEAN DEFAULT FALSE,
    ike_mode              INTEGER NOT NULL DEFAULT 4,
    ike_nat_t_enabled     BOOLEAN NOT NULL DEFAULT TRUE,
    ike_nat_t_keep_alive  INTEGER NOT NULL DEFAULT 20,
    ike_ka_enabled        BOOLEAN NOT NULL DEFAULT FALSE,
    ike_ka_interval       INTEGER NOT NULL DEFAULT 30,
    ike_ka_max            INTEGER NOT NULL DEFAULT 5,
    ike_dpd_enabled       BOOLEAN NOT NULL DEFAULT TRUE,
    ike_dpd_interval      INTEGER NOT NULL DEFAULT 20,
    ike_dpd_max           INTEGER NOT NULL DEFAULT 5,
    ike_auth_algm         INTEGER NOT NULL DEFAULT 4,
    ike_encryp_algm       INTEGER NOT NULL DEFAULT 7,
    ike_encryp_key_len    INTEGER NOT NULL DEFAULT 32,
    ike_lifetime          INTEGER NOT NULL DEFAULT 480,
    ike_time_unit         INTEGER NOT NULL DEFAULT 2,
    ike_dh_group          INTEGER NOT NULL DEFAULT 5,
    ipsec_pfs             BOOLEAN NOT NULL DEFAULT TRUE,
    ipsec_dh_group        INTEGER NOT NULL DEFAULT 5,
    ipsec_auth_algm       INTEGER NOT NULL DEFAULT 4,
    ipsec_encryp_algm     INTEGER NOT NULL DEFAULT 7,
    ipsec_encryp_key_len  INTEGER NOT NULL DEFAULT 32,
    ipsec_lifetime        INTEGER NOT NULL DEFAULT 480,
    ipsec_time_unit       INTEGER NOT NULL DEFAULT 2,
    ipsec_life_length     INTEGER NOT NULL DEFAULT 0,
    PRIMARY KEY (id)
);

-- recommended default settings
INSERT INTO vpn_security_templates (id) VALUES (0);

-- contains entries for gateway devices associated with a managed VPN
CREATE TABLE vpn_gateways (
    id                    INTEGER NOT NULL DEFAULT NEXTVAL('vpn_gateway_seq'),
    gateway_type          INTEGER NOT NULL,  -- 0 = hub, 1 = spoke
    managed_vpn_id        INTEGER NOT NULL REFERENCES managed_vpns(id) ON DELETE CASCADE,
    device_id             INTEGER NOT NULL,
    seq_order             INTEGER NOT NULL DEFAULT 0,
    name                  TEXT NOT NULL DEFAULT '',
    policy                INTEGER NOT NULL DEFAULT 1,
    update_time           TIMESTAMP WITHOUT TIME ZONE NOT NULL
                          DEFAULT date_trunc('second'::text, timezone('utc'::text, clock_timestamp())),
    virtual_ip            TEXT NOT NULL DEFAULT '',
    PRIMARY KEY (id)
);

-- hubs have no additional columns
CREATE TABLE vpn_hub_gateways (
) INHERITS (vpn_gateways);

-- spoke gateways have additional columns
CREATE TABLE vpn_spoke_gateways (
    auto_start            BOOLEAN NOT NULL DEFAULT TRUE,
    tunnel_mode           INTEGER NOT NULL DEFAULT 1,
    tunnel_switching      BOOLEAN NOT NULL DEFAULT FALSE
) INHERITS (vpn_gateways);

-- vpn resources
CREATE TABLE vpn_resources (
    id                    INTEGER NOT NULL DEFAULT NEXTVAL('wg_vpn_seq'),
    resource_type         INTEGER NOT NULL, -- 0 = traditional, 1 = vif
    gateway_id            INTEGER NOT NULL,
    ip_address            TEXT NOT NULL,
    PRIMARY KEY (id)
);

CREATE TABLE vpn_traditional_resources (
    direction             INTEGER NOT NULL DEFAULT 0,
    broadcast_enabled     BOOLEAN NOT NULL DEFAULT FALSE,
    nat_enabled           BOOLEAN NOT NULL DEFAULT FALSE,
    nat_address           TEXT DEFAULT NULL
) INHERITS (vpn_resources);

CREATE TABLE vpn_vif_resources (
    metric                INTEGER NOT NULL DEFAULT 1
) INHERITS (vpn_resources);

-- ties gateway types to interface name
CREATE TABLE vpn_endpoints (
    id                    INTEGER NOT NULL DEFAULT NEXTVAL('wg_vpn_seq'),
    gateway_id            INTEGER NOT NULL,
    ifname                TEXT,
    PRIMARY KEY (id)
);

-- ties hub and spoke together
CREATE TABLE vpn_tunnels (
    id                    INTEGER NOT NULL DEFAULT NEXTVAL('wg_vpn_seq'),
    spoke_resource_id     INTEGER NOT NULL,
    hub_resource_id       INTEGER NOT NULL,
    PRIMARY KEY (id)
);

CREATE TABLE policy_counter_stats (
    health_id       INTEGER NOT NULL,
    name            TEXT NOT NULL,
    bytes           BIGINT NOT NULL DEFAULT 0,
    packets         BIGINT NOT NULL DEFAULT 0,
    connections     INTEGER NOT NULL DEFAULT 0,
    discards        INTEGER NOT NULL DEFAULT 0,
    PRIMARY KEY (health_id, name)
);
-- rule to clean up policy counter data when health record is removed
CREATE RULE cleanup_policy_counter_stats AS
    ON DELETE TO device_health
    DO DELETE FROM policy_counter_stats WHERE health_id = old.id;

