-- This file contains the SQL statements necessary to create the 'auth'
-- tables in the userdb schema for Dimension. It is intended to be run
-- by the web UI (or other) when it initially starts operation.

-- Create a new schema for the user database
CREATE SCHEMA userdb;

-- look in userdb first, then the public schema
SET search_path TO userdb, public;

-- User authentication servers
CREATE SEQUENCE auth_server_seq MINVALUE 1;
CREATE TABLE auth_servers (
    id              INTEGER NOT NULL DEFAULT NEXTVAL('auth_server_seq'),
    auth_domain     TEXT,
    validate_cert   BOOLEAN DEFAULT FALSE,
    enabled         BOOLEAN DEFAULT FALSE,
    server_type     TEXT NOT NULL DEFAULT 'AD',
    server_addr     TEXT DEFAULT NULL,
    secret          TEXT DEFAULT NULL,
    port            INTEGER DEFAULT NULL,
    PRIMARY KEY(id)
);

CREATE TABLE user_config (
    setting         TEXT NOT NULL,
    value           TEXT NOT NULL,
    PRIMARY KEY(setting)
);

INSERT INTO user_config (setting, value) SELECT 'auth_server_timeout', '10'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'auth_server_timeout');
INSERT INTO user_config (setting, value) SELECT 'auth_server_group_filter', '11'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'auth_server_group_filter');
INSERT INTO user_config (setting, value) SELECT 'auth_server_retries', '3'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'auth_server_retries');

INSERT INTO user_config (setting, value) SELECT 'lockout_enable', 'false'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'lockout_enable');
INSERT INTO user_config (setting, value) SELECT 'lockout_nsteps', '5'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'lockout_nsteps');
INSERT INTO user_config (setting, value) SELECT 'lockout_stepsize', '5'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'lockout_stepsize');
INSERT INTO user_config (setting, value) SELECT 'lockout_duration', '5'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'lockout_duration');

INSERT INTO user_config (setting, value) SELECT 'network_acl_enable', 'true'
    WHERE NOT EXISTS (SELECT 1 FROM user_config WHERE setting = 'network_acl_enable');

-- User details
CREATE SEQUENCE users_seq MINVALUE 1;
CREATE TABLE users (
    id               INTEGER NOT NULL DEFAULT NEXTVAL('users_seq'),
    login            TEXT NOT NULL,
    name             TEXT,
    password         TEXT,       -- sha2 with salt
    email            TEXT,
    user_type        TEXT,       -- local/AD/RADIUS user/group
    failed_logins    INTEGER NOT NULL DEFAULT 0,
    lockedout_until  TIMESTAMP NOT NULL DEFAULT TIMEZONE('UTC', NOW()),
    password_control INTEGER NOT NULL DEFAULT 0, -- Local non-admin user only.
                                                 -- Permission to change password.
                                                 -- 0/1/2: no/yes/must change
    PRIMARY KEY(id)
);
CREATE UNIQUE INDEX users_uk_idx ON users(lower(login), user_type);

-- add admin user
INSERT INTO users (id, login) VALUES (0, 'admin');

-- User group membership
CREATE TABLE user_groups (
    user_group_id  INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    user_id        INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE
);

-- UI user/network ACL
CREATE TABLE network_access (
    user_id     INTEGER NOT NULL,
    network     TEXT NOT NULL
);

-- Roles that a user or user group can play
CREATE TABLE roles (
    id          INTEGER NOT NULL,
    name        TEXT NOT NULL,
    locked      BOOLEAN DEFAULT FALSE,    -- predefined roles cannot be altered
    PRIMARY KEY(id)
);

--add roles
INSERT INTO roles SELECT 1, 'Super Administrator', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=1);
INSERT INTO roles SELECT 2, 'Report Administrator', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=2);
INSERT INTO roles SELECT 3, 'Device Administrator', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=3);
INSERT INTO roles SELECT 4, 'View Logs', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=4);
INSERT INTO roles SELECT 5, 'View Reports', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=5);
--dimension 2.0 roles
INSERT INTO roles SELECT 6, 'Global Administrator', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=6);
INSERT INTO roles SELECT 7, 'Global Monitor', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=7);
INSERT INTO roles SELECT 8, 'View All Logs', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=8);
INSERT INTO roles SELECT 9, 'View All Reports', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=9);
INSERT INTO roles SELECT 10, 'Device Monitor', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=10);
INSERT INTO roles SELECT 11, 'VPN Administrator', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=11);
INSERT INTO roles SELECT 12, 'VPN Monitor', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=12);
--dimension 2.1 role
INSERT INTO roles SELECT 15, 'Anonymization Officer', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=15);

--dimension 2.1.1 role
INSERT INTO roles SELECT 16, 'Dimension Administrator', true WHERE
    NOT EXISTS (SELECT 1 FROM roles WHERE id=16);

-- Role policies that can be assigned to a user or user group
CREATE SEQUENCE role_policy_seq MINVALUE 1;
CREATE TABLE role_policy (
    id              INTEGER NOT NULL DEFAULT NEXTVAL('role_policy_seq'),
    user_id         INTEGER NOT NULL DEFAULT 0 REFERENCES users(id) ON DELETE CASCADE,
    role_id         INTEGER NOT NULL REFERENCES roles(id) ON DELETE CASCADE,
    PRIMARY KEY(id)
);

-- Devices assigned to a particular role policy
CREATE TABLE role_policy_devices (
    role_policy_id     INTEGER NOT NULL REFERENCES role_policy(id) ON DELETE CASCADE,
    device_id          INTEGER NOT NULL
);

-- Device groups assigned to a particular role policy
CREATE TABLE role_policy_groups (
    role_policy_id     INTEGER NOT NULL REFERENCES role_policy(id) ON DELETE CASCADE,
    group_id           INTEGER NOT NULL REFERENCES group_name(id) ON DELETE CASCADE
);

-- Device clusters assigned to a particular role policy
CREATE TABLE role_policy_clusters (
    role_policy_id     INTEGER NOT NULL REFERENCES role_policy(id) ON DELETE CASCADE,
    cluster_id         INTEGER NOT NULL REFERENCES g_clusterinfo(id) ON DELETE CASCADE
);

-- Operations that can be performed.
-- This includes data for the UI: common name for the operation and its location.
CREATE TABLE operations (
    id          INTEGER NOT NULL,
    section     SMALLINT NOT NULL,   -- UI aid, 1=Management Server, 2=Folders, 3=device
    name        TEXT NOT NULL,
    expand      BOOLEAN,             -- whether or not this operation expands to rw/ro
    PRIMARY KEY(id)
);

-- add operations
INSERT INTO operations SELECT 0, 1, 'All operations', false WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=0);
INSERT INTO operations SELECT 1, 1, 'Manage Users, Groups and Role Policies', true WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=1);
INSERT INTO operations SELECT 2, 1, 'All Devices', true WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=2);
INSERT INTO operations SELECT 9, 1, 'View All Reports', true WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=9);
INSERT INTO operations SELECT 10, 1, 'View All Logs', false WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=10);
INSERT INTO operations SELECT 12, 1, 'Manage device groups', true WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=12);
INSERT INTO operations SELECT 104, 3, 'View Logs', false WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=104);
INSERT INTO operations SELECT 105, 3, 'View Reports', false WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=105);
-- dimension 2.0 operations
INSERT INTO operations SELECT 6, 1, 'Manage contact list', true WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=6);
INSERT INTO operations SELECT 110, 3, 'Manage VPN', true WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=110);
INSERT INTO operations SELECT 200, 3, 'Manage Device', true WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=200);
-- dimension 2.1 operations
INSERT INTO operations SELECT 15, 1, 'Manage Anonymization Officer', false WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=15);
-- dimension 2.1.1 operations
INSERT INTO operations SELECT 16, 1, 'Manage Dimension', false WHERE
    NOT EXISTS (SELECT 1 FROM operations WHERE id=16);

-- Reference table which ties the various permissions to operations
CREATE TABLE permissions (
    id              INTEGER NOT NULL,
    operation_id    INTEGER NOT NULL REFERENCES operations(id) ON DELETE CASCADE,
    permission      SMALLINT NOT NULL DEFAULT 0,  -- 0 = none, 1 = read only, 2 = read/write
    PRIMARY KEY(id)
);

-- add permissions
-- operation: 'All operations', permission: view or read only
INSERT INTO permissions SELECT 1, 0, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=1);
-- operation: 'All operations', permission: allow or read/write
INSERT INTO permissions SELECT 2, 0, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=2);

-- operation: 'Manage Users, Groups and Role Policies', permission: view or read only
INSERT INTO permissions SELECT 3, 1, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=3);
-- operation: 'Manage Users, Groups and Role Policies', permission: allow or read/write
INSERT INTO permissions SELECT 4, 1, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=4);

-- operation: 'All Devices', permission: view or read only
INSERT INTO permissions SELECT 5, 2, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=5);
-- operation: 'All Devices', permission: allow or read/write
INSERT INTO permissions SELECT 6, 2, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=6);

-- operation: 'View All Reports', permission: view or read only
INSERT INTO permissions SELECT 7, 9, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=7);
-- operation: 'View All Reports', permission: allow or read/write
INSERT INTO permissions SELECT 8, 9, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=8);

-- operation: 'View All Logs', permission: view or read only
INSERT INTO permissions SELECT 9, 10, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=9);
-- operation: 'View All Logs', permission: allow or read/write
INSERT INTO permissions SELECT 10, 10, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=10);

-- operation: 'Manage device groups', permission: view or read only
INSERT INTO permissions SELECT 11, 12, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=11);
-- operation: 'Manage device groups', permission: allow or read/write
INSERT INTO permissions SELECT 12, 12, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=12);

-- operation: 'View Logs', permission: view or read only
INSERT INTO permissions SELECT 13, 104, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=13);
-- operation: 'View Logs', permission: allow or read/write
INSERT INTO permissions SELECT 14, 104, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=14);

-- operation: 'View Reports', permission: view or read only
INSERT INTO permissions SELECT 15, 105, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=15);
-- operation: 'View Reports', permission: allow or read/write
INSERT INTO permissions SELECT 16, 105, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=16);

--dimension 2.0 permissions
-- operation: 'Manage contact list', permission: view or read only
INSERT INTO permissions SELECT 17, 6, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=17);
-- operation: 'Manage contact list', permission: allow or read/write
INSERT INTO permissions SELECT 18, 6, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=18);

-- operation: 'Manage VPN', permission: view or read only
INSERT INTO permissions SELECT 19, 110, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=19);
-- operation: 'Manage VPN', permission: allow or read/write
INSERT INTO permissions SELECT 20, 110, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=20);

-- operation: 'Manage Device', permission: view or read only
INSERT INTO permissions SELECT 21, 200, 1 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=21);
-- operation: 'Manage Device', permission: allow or read/write
INSERT INTO permissions SELECT 22, 200, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=22);

--dimension 2.1 permissions
-- operation: 'Manage Anonymization Officer', permission: allow or read/write
INSERT INTO permissions SELECT 23, 15, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=23);

--dimension 2.1.1 permissions
-- operation: 'Manage Dimension', permission: allow or read/write
INSERT INTO permissions SELECT 24, 16, 2 WHERE
    NOT EXISTS (SELECT 1 FROM permissions WHERE id=24);

-- Permissions assigned to roles
CREATE TABLE role_permissions (
    role_id          INTEGER NOT NULL REFERENCES roles(id) ON DELETE CASCADE,
    permission_id    INTEGER NOT NULL REFERENCES permissions(id) ON DELETE CASCADE
);

-- add role_permissions
-- role: 'Super Administrator', (operation: 0, permission: 2) - All operations / Allow
INSERT INTO role_permissions SELECT 1, 2 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=1 and permission_id=2);

-- role: 'Super Administrator', (operation: 1, permission: 2) - Manage Users, Groups and Role Policies / Allow
INSERT INTO role_permissions SELECT 1, 4 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=1 and permission_id=4);

-- role: 'Report Administrator', (operation: 9, permission: 2) - View All Reports / Allow
INSERT INTO role_permissions SELECT 2, 8 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=2 and permission_id=8);

-- role: 'Report Administrator', (operation: 10, permission: 2) - View All Logs / Allow
INSERT INTO role_permissions SELECT 2, 10 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=2 and permission_id=10);

-- role: 'Report Administrator', (operation: 12, permission: 2) - Manage device groups / Allow
INSERT INTO role_permissions SELECT 2, 12 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=2 and permission_id=12);

-- role: 'Device Administrator', (operation: 200, permission: 2) - Manage Device / Allow
INSERT INTO role_permissions SELECT 3, 22 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=3 and permission_id=22);

-- role: 'Device Administrator', (operation: 6, permission: 1) - Manage contact list / View
INSERT INTO role_permissions SELECT 3, 17 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=3 and permission_id=17);

-- role: 'View logs' , (operation: 104, permission: 2) - View Logs / Allow
INSERT INTO role_permissions SELECT 4, 14 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=4 and permission_id=14);

-- role: 'View logs', (operation: 12, permission: 1) - Manage device groups / View
INSERT INTO role_permissions SELECT 4, 11 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=4 and permission_id=11);

-- role: 'View reports', (operation: 105, permission: 2) - View Reports / Allow
INSERT INTO role_permissions SELECT 5, 16 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=5 and permission_id=16);

-- role: 'View reports', (operation: 12, permission: 1) - Manage device groups / View
INSERT INTO role_permissions SELECT 5, 11 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=5 and permission_id=11);

-- role: 'Global Administrator', (operation: 2, permission: 2) - All Devices / Allow
INSERT INTO role_permissions SELECT 6, 6 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=6 and permission_id=6);

-- role: 'Global Administrator', (operation: 6, permission: 2) - Manage contact list / Allow
INSERT INTO role_permissions SELECT 6, 18 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=6 and permission_id=18);

-- role: 'Global Administrator', (operation: 9, permission: 2) - View All Reports / Allow
INSERT INTO role_permissions SELECT 6, 8 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=6 and permission_id=8);

-- role: 'Global Administrator', (operation: 10, permission: 2) - View All Logs / Allow
INSERT INTO role_permissions SELECT 6, 10 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=6 and permission_id=10);

-- role: 'Global Administrator', (operation: 12, permission: 2) - Manage device groups / Allow
INSERT INTO role_permissions SELECT 6, 12 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=6 and permission_id=12);

-- role: 'Global Monitor', (operation: 2, permission: 1) - All Devices / View
INSERT INTO role_permissions SELECT 7, 5 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=7 and permission_id=5);

-- role: 'Global Monitor', (operation: 6, permission: 1) - Manage contact list / View
INSERT INTO role_permissions SELECT 7, 17 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=7 and permission_id=17);

-- role: 'Global Monitor', (operation: 9, permission: 2) - View All Reports / View
INSERT INTO role_permissions SELECT 7, 8 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=7 and permission_id=8);

-- role: 'Global Monitor', (operation: 10, permission: 2) - View All Logs / View
INSERT INTO role_permissions SELECT 7, 10 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=7 and permission_id=10);

-- role: 'Global Monitor', (operation: 12, permission: 1) - Manage device groups / View
INSERT INTO role_permissions SELECT 7, 11 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=7 and permission_id=11);

-- role: 'View all logs', (operation: 10, permission: 2) - View All Logs / Allow
INSERT INTO role_permissions SELECT 8, 10 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=8 and permission_id=10);

-- role: 'View all logs', (operation: 12, permission: 1) - Manage device groups / View
INSERT INTO role_permissions SELECT 8, 11 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=8 and permission_id=11);

-- role: 'View all reports', (operation: 9, permission: 2) - View All Reports / Allow
INSERT INTO role_permissions SELECT 9, 8 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=9 and permission_id=8);

-- role: 'View all reports', (operation: 12, permission: 1) - Manage device groups / View
INSERT INTO role_permissions SELECT 9, 11 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=9 and permission_id=11);

-- role: 'Device Monitor', (operation: 6, permission: 1) - Manage Device / View
INSERT INTO role_permissions SELECT 10, 21 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=10 and permission_id=21);

-- role: 'Device Monitor', (operation: 200, permission: 1) - Manage contact list / View
INSERT INTO role_permissions SELECT 10, 17 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=10 and permission_id=17);

-- role: 'VPN Administrator', (operation: 110, permission: 2) - Manage VPN / Allow
INSERT INTO role_permissions SELECT 11, 20 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=11 and permission_id=20);

-- role: 'VPN Administrator', (operation: 6, permission: 1) - Manage contact list / View
INSERT INTO role_permissions SELECT 11, 17 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=11 and permission_id=21);

-- role: 'VPN Administrator', (operation: 200, permission: 1) - Manage VPN / View
INSERT INTO role_permissions SELECT 11, 21 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=11 and permission_id=21);

-- role: 'VPN Monitor', (operation: 110, permission: 1) - Manage VPN / View
INSERT INTO role_permissions SELECT 12, 19 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=12 and permission_id=19);

-- role: 'VPN Monitor', (operation: 6, permission: 1) - Manage contact list / View
INSERT INTO role_permissions SELECT 12, 17 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=12 and permission_id=17);

-- role: 'VPN Monitor', (operation: 200, permission: 1) - Manage VPN / View
INSERT INTO role_permissions SELECT 12, 21 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=12 and permission_id=21);

-- Dimension 2.1: add role_permissions
-- role: 'Anonymization Officer', (operation: 15, permission: 2) - Manage Anonymization Officer
INSERT INTO role_permissions SELECT 15, 23 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=15 and permission_id=23);

-- Dimension 2.1.1: add role_permissions
-- role: 'Dimension Administrator', (operation: 16, permission: 2) - Manage Dimension
INSERT INTO role_permissions SELECT 16, 24 WHERE
    NOT EXISTS (SELECT 1 FROM role_permissions WHERE role_id=16 and permission_id=24);

-- fix access permissions to this schema
GRANT ALL ON ALL TABLES IN SCHEMA userdb TO PUBLIC;


