#!/bin/sh

# wg_apache_tune.sh - Script to rewrite the runtime tuning
#    configuration of wsserver's Apache. This will adjust the number 
#    of threads based on the memory size using the same algorithm as
#    log collector does to compute the maximum number of clients. 
#    (see //gsm/development/wgagent/mainline/src/wgpts/main.c)

PROG=$0

# Constants from wlcollector/service.h
WGPTS_BYTES_PER_THREAD=1048576*2
WGPTS_ALLOWED_MEMORY_PCT=25

# Additional threads for servicing UI
# (note this should be a multiple of the serverlimit)
extra_threads=64

# File handles per web UI thread. This includes a socket connection
# for the browser, a handle for session data (database or file) plus 2
# handles reserved for other transient file I/O.
files_per_ui_thread=4

# File handles per DXCP connection. This includes socket connections for the incoming
# and proxied connections
files_per_dxcp=3

# File handles in addition to those required to service DXCP and web UI handlers
extra_open_files=256

# Number of child servers
serverlimit=1

err_exit() {
    local code=$1
    local msg="$2"
    echo $PROG: error $code: $msg
    exit $code
}

# Replace an old file with a new one if it changed
install_updated() {
    local oldfile=$1
    local newfile=$2

    if [ ! -s $newfile ] ; then
        echo "$newfile does not exist"
        return
    fi
    
    chown ${WG_ADMIN_USER}:${WG_ADMIN_GROUP} $newfile
    chmod 660 $newfile
    if [ -s $oldfile ] ; then
        # check if changed
        if diff -q $oldfile $newfile > /dev/null 2>&1 ; then
            echo "$0: $oldfile unchanged"
            return
        fi
    fi
    
    # file either doesn't exist or has changed, so install it!
    if mv $newfile $oldfile ; then
        echo "$0: $oldfile updated"
    else
        err_exit -6 "Unable to install $oldfile"
    fi
}

if [ -r /etc/default/wg_system ] ; then 
    . /etc/default/wg_system
else
    err_exit -1 "unable to read /etc/default/wg_system"
fi

# Fetch system memory in KB
if [ -r /proc/meminfo ] ; then
    memkb=$(($(grep MemTotal /proc/meminfo|awk '{print $2}') ))
else
    err_exit -2 "unable to read /proc/meminfo"
fi

# Check postgres.ini for external database and adjust allowed memory
allowed_mem_pct=${WGPTS_ALLOWED_MEMORY_PCT}
if grep external_db ${WG_ETCDIR}/postgres/postgres.ini 2>/dev/null| grep -q 1 ; then 
    # external db used, so double allowed memory for threads
    allowed_mem_pct=$((${WGPTS_ALLOWED_MEMORY_PCT} * 2))
fi

# Calculate number of allowed threads
max_threads=$(($memkb * 1024 * $allowed_mem_pct / 100 / (${WGPTS_BYTES_PER_THREAD}) + $extra_threads))
# Round up when calculating threads per child
threads_per_child=$(( ($max_threads + $serverlimit - 1)/ $serverlimit))
# Maximum number of websocket connections allowed through proxy per child
ws_per_child=$(($threads_per_child - ($extra_threads / $serverlimit) ))
# ensure max_threads is a multiple of threads_per_child
max_threads=$(($threads_per_child * $serverlimit))
# max spare threads needs to be > min_spare_threads + threads per child
max_spare_threads=$(($threads_per_child + (2 * $extra_threads) ))
# max dxcp clients
max_dxcp_clients=$(($max_threads - $extra_threads))
# max open files
max_open_files=$(($max_dxcp_clients * $files_per_dxcp + $extra_threads * $files_per_ui_thread + $extra_open_files))

CONF_DIR=${WG_ETCDIR}/wsserver/conf
TUNING_CONF=${CONF_DIR}/tuning.conf
TUNING_CONF_TMP=${TUNING_CONF}.tmp
WSSERVER_DEFAULTS=/etc/default/wsserver
WSSERVER_DEFAULTS_TMP=${WSSERVER_DEFAULTS}.tmp

DXCP_DIR=${WG_ETCDIR}/wsserver/dxcp
DXCP_JSON=${DXCP_DIR}/dxcp.json
DXCP_JSON_TMP=${DXCP_JSON}.tmp

# clean up on exit
trap "rm -f ${TUNING_CONF_TMP} ${DXCP_JSON_TMP} ${WSSERVER_DEFAULTS_TMP}" 0 1 2 3 15

# Create required directories
if [ ! -d ${CONF_DIR} ] ; then
    mkdir -p ${CONF_DIR} || err_exit -3 "Unable to create ${CONF_DIR}"
    chown ${WG_ADMIN_USER}:${WG_ADMIN_GROUP} ${CONF_DIR} ${CONF_DIR}/..
    chmod 770 ${CONF_DIR} ${CONF_DIR}/..
fi
if [ ! -d ${DXCP_DIR} ] ; then
    mkdir -p ${DXCP_DIR} || err_exit -3 "Unable to create ${DXCP_DIR}"
    chown ${WG_ADMIN_USER}:${WG_ADMIN_GROUP} ${DXCP_DIR}
    chmod 770 ${DXCP_DIR}
fi

# Create new Apache tuning.conf
rm -f ${TUNING_CONF_TMP} || err_exit -4 "Unable to remove ${TUNING_CONF_TMP}"
cat <<EOF > ${TUNING_CONF_TMP} || err_exit -5 "Unable to create ${TUNING_CONF_TMP}"
ServerLimit                $serverlimit
MaxClients                 $max_threads
MinSpareThreads            $extra_threads
MaxSpareThreads            $max_spare_threads
ThreadsPerChild            $threads_per_child
ThreadLimit                $max_threads
Define max_ws_conns        $ws_per_child
EOF
install_updated ${TUNING_CONF} ${TUNING_CONF_TMP}

# Create dxcp.json if it doesn't exist, update it if it does
if [ -s ${DXCP_JSON} ] ; then
    rm -f ${DXCP_JSON_TMP} || err_exit -4 "Unable to remove ${DXCP_JSON_TMP}"
    sed -r -e "/max_clients/ s/(max_clients\": *)[0-9]+/\1${max_dxcp_clients}/" ${DXCP_JSON} > ${DXCP_JSON_TMP}
else
    cat <<EOF2 > ${DXCP_JSON_TMP} || err_exit -6 "Unable to create ${DXCP_JSON_TMP}"
{
    "max_clients": $max_dxcp_clients
}
EOF2
fi
install_updated ${DXCP_JSON} ${DXCP_JSON_TMP}

# Create or update wsserver default settings file to set ulimit, if needed
if [ $max_open_files -gt 1024 ] ; then
    echo "ulimit -n $max_open_files || echo 'Unable to set ulimit'" > ${WSSERVER_DEFAULTS_TMP}
fi
if [ -s ${WSSERVER_DEFAULTS} ] ; then
    grep -v "^ulimit" ${WSSERVER_DEFAULTS} >> ${WSSERVER_DEFAULTS_TMP}
fi
if [ -s ${WSSERVER_DEFAULTS_TMP} ] ; then
    install_updated ${WSSERVER_DEFAULTS} ${WSSERVER_DEFAULTS_TMP}
else
    rm -f ${WSSERVER_DEFAULTS}
fi