#!/bin/sh
# wg-vm-detect.sh - Shell script to detect virtual machine platform.

_log_error() {
    logger -s -p local0.err -t "wg_vm_detect" "ERROR: ${1}"
}

_log_notice() {
    logger -s -p local0.notice -t "wg_vm_detect" "NOTICE: ${1}"
}

_configure_ssh_add_user() {
    "$WG_BINDIR/python" - "$@" <<EOF
import json, sys
user = sys.argv[1]
input = open(sys.argv[2], 'r')
output = open(sys.argv[3], 'w')
config = json.load(input)
if 'enabled' in config and config['enabled'] is True and not 'users' in config:
    config['users'] = ['$WG_ADMIN_USER']
if not 'users' in config:
    config['users'] = []
if not user in config['users']:
    config['users'].append(user)
config['enabled'] = True
string = json.dumps(config, sort_keys=True, indent=4)
output.write(string + '\n')
EOF
}

_configure_ssh_remove_user() {
    "$WG_BINDIR/python" - "$@" <<EOF
import json, sys
user = sys.argv[1]
input = open(sys.argv[2], 'r')
output = open(sys.argv[3], 'w')
config = json.load(input)
if 'users' in config:
    if user in config['users']:
        config['users'].remove(user)
    if len(config['users']) == 0:
        config['enabled'] = False
string = json.dumps(config, sort_keys=True, indent=4)
output.write(string + '\n')
EOF
}

_configure_system_hostname() {
    _log_notice "Writing system metadata to $2"

    if _configure_system_value hostname "$@"; then
        chmod go+r,g+w "$3"
        _save_file "$3" "$2" "$WG_ADMIN_USER":"$WG_ADMIN_GROUP"
        _log_notice "Wrote system metadata to $2"
    else
        _log_error "Unable to update system metadata"
    fi
}

_configure_system_value() {
    "$WG_BINDIR/python" - "$@" <<EOF
import json, sys
key = sys.argv[1]
value = sys.argv[2]
input = open(sys.argv[3], 'r')
output = open(sys.argv[4], 'w')
config = json.load(input)
config[key] = value
string = json.dumps(config, sort_keys=True, indent=4)
output.write(string + '\n')
EOF
}

_create_temporary_file() {
    local tmp_file=`mktemp`
    eval "$1"="$tmp_file"
    TMP_FILES="$tmp_file $TMP_FILES"
}

_delete_temporary_files() {
    /bin/rm -f $TMP_FILES
}

_print_platform_info() {
    echo    "{"
    echo    "    \"vm-vendor\": \"$WG_VM_VENDOR\","
    echo -n "    \"vm-type\": \"$WG_VM_TYPE\""

    if [ "$WG_VM_VENDOR" = Amazon-EC2 ]; then

        EC2_AVAILABILITY_ZONE=`$WGET $METADATA/placement/availability-zone`
        EC2_LOCAL_IPV4=`$WGET $METADATA/local-ipv4`

        echo    ","
        echo    "    \"vm-data\": {"
        echo    "        \"availability-zone\": \"$EC2_AVAILABILITY_ZONE\","
        echo    "        \"instance-id\": \"$EC2_INSTANCE_ID\","
        echo    "        \"local-hostname\": \"$EC2_LOCAL_HOSTNAME\","
        echo    "        \"local-ipv4\": \"$EC2_LOCAL_IPV4\","
        echo -n "        \"network-interfaces\": ["

        EC2_NETWORK_MACS=`$WGET $METADATA/network/interfaces/macs`
        INTERFACE_COUNT=0

        for EC2_NETWORK_MAC in $EC2_NETWORK_MACS; do

            EC2_NETWORK_MAC=`echo $EC2_NETWORK_MAC | sed 's/\/$//g'`
            EC2_DEVICE_NUMBER=`$WGET $METADATA/network/interfaces/macs/$EC2_NETWORK_MAC/device-number`
            EC2_LOCAL_HOSTNAMES=`$WGET $METADATA/network/interfaces/macs/$EC2_NETWORK_MAC/local-hostname`
            EC2_LOCAL_IPV4S=`$WGET $METADATA/network/interfaces/macs/$EC2_NETWORK_MAC/local-ipv4s`
            EC2_PUBLIC_HOSTNAMES=`$WGET $METADATA/network/interfaces/macs/$EC2_NETWORK_MAC/public-hostname || true`
            EC2_PUBLIC_IPV4S=`$WGET $METADATA/network/interfaces/macs/$EC2_NETWORK_MAC/public-ipv4s || true`
            EC2_VPC_ID=`$WGET $METADATA/network/interfaces/macs/$EC2_NETWORK_MAC/vpc-id || true`

            if [ $INTERFACE_COUNT -gt 0 ]; then
                echo    ","
            else
                echo
            fi

            echo    "            {"
            echo    "                \"device-number\": \"$EC2_DEVICE_NUMBER\","
            echo -n "                \"local-hostnames\": ["

            HOSTNAME_COUNT=0

            for EC2_LOCAL_HOSTNAME in $EC2_LOCAL_HOSTNAMES; do

                if [ $HOSTNAME_COUNT -gt 0 ]; then
                    echo    ","
                else
                    echo
                fi

                echo -n "                    \"$EC2_LOCAL_HOSTNAME\""
                HOSTNAME_COUNT=$((HOSTNAME_COUNT + 1))

            done

            echo
            echo    "                ],"
            echo -n "                \"local-ipv4s\": ["

            IPV4_COUNT=0

            for EC2_LOCAL_IPV4 in $EC2_LOCAL_IPV4S; do

                if [ $IPV4_COUNT -gt 0 ]; then
                    echo    ","
                else
                    echo
                fi

                echo -n "                    \"$EC2_LOCAL_IPV4\""
                IPV4_COUNT=$((IPV4_COUNT + 1))

            done

            echo
            echo    "                ],"
            echo    "                \"mac\": \"$EC2_NETWORK_MAC\","
            echo -n "                \"public-hostnames\": ["

            HOSTNAME_COUNT=0

            for EC2_PUBLIC_HOSTNAME in $EC2_PUBLIC_HOSTNAMES; do

                if [ $HOSTNAME_COUNT -gt 0 ]; then
                    echo    ","
                else
                    echo
                fi

                echo -n "                    \"$EC2_PUBLIC_HOSTNAME\""
                HOSTNAME_COUNT=$((HOSTNAME_COUNT + 1))

            done

            echo
            echo    "                ],"
            echo -n "                \"public-ipv4s\": ["

            IPV4_COUNT=0

            for EC2_PUBLIC_IPV4 in $EC2_PUBLIC_IPV4S; do

                if [ $IPV4_COUNT -gt 0 ]; then
                    echo    ","
                else
                    echo
                fi

                echo -n "                    \"$EC2_PUBLIC_IPV4\""
                IPV4_COUNT=$((IPV4_COUNT + 1))

            done

            echo
            echo    "                ],"
            echo    "                \"vpc-id\": \"$EC2_VPC_ID\""
            echo -n "            }"
            INTERFACE_COUNT=$((INTERFACE_COUNT + 1))

        done

        echo
        echo -n "        ]"

        if [ -n "$EC2_OPENSSH_KEY" ]; then
            echo    ","
            echo    "        \"openssh-key\": \"$EC2_OPENSSH_KEY\""
        else
            echo
        fi

        echo    "    }"

    else

        echo

    fi

    echo    "}"
}

_save_file() {
    chown "$3" "$1"
    /usr/bin/cmp -s "$1" "$2" || /bin/mv -f "$1" "$2"
}

_set_platform_info() {
    _log_notice "Writing virtualization platform metadata to $1"

    if _print_platform_info >"$2"; then
        chmod go+r,g+w "$2"
        _save_file "$2" "$1" "$WG_ADMIN_USER":"$WG_ADMIN_GROUP"
        _log_notice "Wrote virtualization platform metadata to $1"
    else
        _log_error "Unable to retrieve one or more metadata items"
    fi
}

add_ssh_user() {
    if _configure_ssh_add_user "$1" "$SSH_FILE" "$TMP_SSH_FILE"; then
        chmod go+r,g+w "$TMP_SSH_FILE"
        _save_file "$TMP_SSH_FILE" "$SSH_FILE" "$WG_ADMIN_USER":"$WG_ADMIN_GROUP"
    fi
}

configure_hostname() {
    if [ "$WG_VM_VENDOR" = Amazon-EC2 ]; then
        _log_notice "Setting hostname to $EC2_LOCAL_HOSTNAME"
        _configure_system_hostname "$EC2_LOCAL_HOSTNAME" "$SYS_FILE" "$TMP_SYS_FILE" 
    fi
}

configure_support_access() {
    if [ "$WG_VM_VENDOR" = Amazon-EC2 ]; then

        _log_notice "Disabling password authentication for $WG_SUPPORT_USER"
        chage -d -1 $WG_SUPPORT_USER
        usermod -L $WG_SUPPORT_USER

        if [ -n "$EC2_OPENSSH_KEY" ]; then
            enable_limited_remote_access
        else
            disable_limited_remote_access
        fi
    elif [ "$WG_VM_VENDOR" = Microsoft-Azure ]; then
        : # $WG_SUPPORT_USER removed during waagent installation
    else
        disable_limited_remote_access
    fi
}

configure_virtualization_platform() {
    if [ "$WG_VM_VENDOR" != Microsoft-Azure ]; then
        _set_platform_info "$VM_HOST_FILE" "$TMP_VM_HOST_FILE"
    fi
}

disable_limited_remote_access() {
    _log_notice "Disabling remote access for $WG_SUPPORT_USER"
    remove_ssh_user "$WG_SUPPORT_USER"

    if [ -f "$KEY_FILE" ]; then
        _log_notice "Uninstalling SSH public keys for $WG_SUPPORT_USER"
        /bin/rm -f "$KEY_FILE"
    fi
}

enable_limited_remote_access() {
    _log_notice "Enabling remote access for $WG_SUPPORT_USER"
    add_ssh_user "$WG_SUPPORT_USER"

    SSH_KEYNAME=`echo "$EC2_OPENSSH_KEY" | awk '{print $NF}'`

    _log_notice "Installing SSH public key \"$SSH_KEYNAME\" for $WG_SUPPORT_USER"
    mkdir -m 700 -p "$KEY_DIR"
    _create_temporary_file TMP_KEY_FILE
    echo "$EC2_OPENSSH_KEY" >$TMP_KEY_FILE
    _save_file "$TMP_KEY_FILE" "$KEY_FILE" "$WG_SUPPORT_USER":"$WG_ADMIN_GROUP"
    chown --reference="$SUPPORT_HOME" -R "$KEY_DIR"
}

remove_ssh_user() {
    if _configure_ssh_remove_user "$1" "$SSH_FILE" "$TMP_SSH_FILE"; then
        chmod go+r,g+w "$TMP_SSH_FILE"
        _save_file "$TMP_SSH_FILE" "$SSH_FILE" "$WG_ADMIN_USER":"$WG_ADMIN_GROUP"
    fi
}
 
#########################
# Execution begins here #
#########################

set -u

_log_notice "Detecting virtualization platform"

. /etc/default/wg_system

SUPPORT_HOME=`getent passwd $WG_SUPPORT_USER | awk -F: '{print $6}'`
TMP_FILES=

trap _delete_temporary_files EXIT HUP INT QUIT TERM

KEY_DIR="$SUPPORT_HOME/.ssh"
KEY_FILE="$KEY_DIR/authorized_keys"
SSH_FILE="$WG_ETCDIR/system/sshd.json"
SYS_FILE="$WG_ETCDIR/system/system.json"
VM_HOST_CONF=/etc/default/wg_vm_host
VM_HOST_FILE=`/bin/readlink -f "$WG_RUNDIR/../vm-host.json"`

mkdir -m 755 -p "$WG_ETCDIR"
mkdir -m 755 -p `dirname "$SSH_FILE"`
_create_temporary_file TMP_SSH_FILE

mkdir -m 755 -p `dirname "$SYS_FILE"`
_create_temporary_file TMP_SYS_FILE 
_create_temporary_file TMP_VM_HOST_CONF

mkdir -m 755 -p `dirname "$VM_HOST_FILE"`
_create_temporary_file TMP_VM_HOST_FILE

if [ -r /etc/default/wg_vm_host ]; then
    . /etc/default/wg_vm_host
else
    # Detect virtualization platform
    WG_VM_TYPE=`lscpu | sed -re 's/:[ ]+/:/' | awk -F: '/Virtualization type:/{print $2}'`
    WG_VM_VENDOR=`lscpu | sed -re 's/:[ ]+/:/' | awk -F: '/Hypervisor vendor:/{print $2}'`

    # Store virtualization platform information in a file
    cat <<EOF >$TMP_VM_HOST_CONF
WG_VM_TYPE=$WG_VM_TYPE
WG_VM_VENDOR=$WG_VM_VENDOR
EOF
    chmod go+r,g+w $TMP_VM_HOST_CONF
    /bin/mv -f $TMP_VM_HOST_CONF $VM_HOST_CONF
fi

if [ "$WG_VM_VENDOR" = Microsoft ]; then
    AZURE_CONF=/etc/default/wg_azure

    if [ -e "$AZURE_CONF" ]; then
        WG_VM_VENDOR=Microsoft-Azure
    fi
elif [ "$WG_VM_VENDOR" = Xen ] || [ "$WG_VM_VENDOR" = KVM ]; then

    METADATA_HOST=169.254.169.254
    METADATA_VERSION=latest

    METADATA=http://$METADATA_HOST/$METADATA_VERSION/meta-data
    WGET="wget -nv -O - -T 1 -t 2"

    EC2_INSTANCE_ID=`$WGET $METADATA/instance-id`

    if [ -n "$EC2_INSTANCE_ID" ]; then
        EC2_LOCAL_HOSTNAME=`$WGET $METADATA/local-hostname`
        EC2_OPENSSH_KEY=`$WGET $METADATA/public-keys/0/openssh-key || true`
        WG_VM_VENDOR=Amazon-EC2
    fi

fi

if [ "$WG_VM_VENDOR" != Microsoft ] && [ "$WG_VM_VENDOR" != Microsoft-Azure ]; then
    # Stop and disable Hyper-V releated services on non-HyperV VM machine
    if [ -f /etc/systemd/system/multi-user.target.wants/hv-kvp-daemon.service ]; then
        systemctl stop hv-kvp-daemon
        systemctl disable hv-kvp-daemon
    fi
fi

_log_notice "Virtualization platform is $WG_VM_VENDOR"
configure_virtualization_platform
configure_support_access
configure_hostname

