#!/bin/sh
# wg-system-init.sh - The shell script for WatchGuard system initialization service

_log_error() {
    logger ${LOGGER_ARGS} -p local0.err "ERROR: ${1}"
}

_log_notice() {
    logger ${LOGGER_ARGS} -p local0.notice "NOTICE: ${1}"
}

_cmp_fname() {
    if [ ! -z "${1}"  -a ! -z "${2}" ] ; then
        fname="`basename ${1}`"
        if [ "$fname" = "${2}" ] ; then
            return 0
        fi
    fi
    return 1
}

_is_system_file(){
    _cmp_fname "${1}" "system.json"
}

_is_interfaces_file(){
    _cmp_fname "${1}" "interfaces.json"
}

_is_interface_file(){
    if [ ! -z "${1}" ] ; then
        if basename "$f" | grep -q '^interface-eth.*\.json$' ; then
            return 0
        fi
    fi
    return 1
}

_is_ntpd_file(){
    _cmp_fname "${1}" "ntpd.json"
}

_is_resolv_file(){
    _cmp_fname "${1}" "resolv.json"
}

_is_sshd_file(){
    _cmp_fname "${1}" "sshd.json"
}

_is_routes_file(){
    _cmp_fname "${1}" "routes.json"
}

_process_file() {
    /opt/watchguard/dimension/bin/wg_event.sh config-change ${1} Modified
}

. /etc/default/wg_system
    
WG_SYSDIR=${WG_ETCDIR}/system

# Detect virtualization platform
VM_TYPE=`lscpu | sed -re 's/:[ ]+/:/' | awk -F: '/Virtualization type:/{print $2}'`
VM_VENDOR=`lscpu | sed -re 's/:[ ]+/:/' | awk -F: '/Hypervisor vendor:/{print $2}'`

# Store virtualization platform information in a file
VM_HOST=/etc/default/wg_vm_host
VM_HOST_TMP=`mktemp $VM_HOST.XXXXXX`
cat <<-EOF >$VM_HOST_TMP
WG_VM_TYPE=$VM_TYPE
WG_VM_VENDOR=$VM_VENDOR
EOF
chmod go+r,g+w $VM_HOST_TMP
/bin/mv -f $VM_HOST_TMP $VM_HOST
    
# re-tune Apache config if needed
if [ -r ${WG_ETCDIR}/wsserver/conf/tuning.conf -a -x ${WG_BINDIR}/wg_apache_tune.sh ] ; then
    msg=`${WG_BINDIR}/wg_apache_tune.sh`
    _log_notice "$msg"
fi

if [ -d "${WG_VARDIR}/system/task-status" ]; then
    # force any "running" system task status files to "failed" on boot
    for f in $(grep -l -R --include="*.json" '"state"\s*:\s*"running"' "${WG_VARDIR}/system/task-status/") ; do
        sed -i -e 's/"state"\s*:\s*"running"/"state":"failed","failure-reason":"Aborted by system reboot"/g' "${f}"
    done
fi

if [ -d ${WG_SYSDIR} ] ; then
    # XXX: Create run directory for Dimension
    /opt/watchguard/dimension/bin/wg-run-dir.sh

    # Always generate the interfaces.json file.
    ethlist=`ifconfig -a | grep "^eth" | awk -F ':' 'BEGIN {printf "[ "} {printf "\"%s\",",$1} END {printf "]\n"}'i|sed -e s'/,]$/ ]/'`
    rm -f ${WG_SYSDIR}/interfaces.json
    echo "$ethlist" > ${WG_SYSDIR}/interfaces.json
    chown --reference=${WG_SYSDIR} ${WG_SYSDIR}/interfaces.json
    chmod 444 ${WG_SYSDIR}/interfaces.json

    if [ ! -f "${WG_SYSDIR}/system.json" ]; then
        # The system.json file is missing. This is generally the case if
        # we are booting up for the first time. 
        _log_notice "Emitting event to set system configuration to default"
        # XXX: For Systemd, no need to trigger the wg-system-reset.service, just run the shell script directly.
        # We still keep wg-system-reset.service as it is since some applications may use it to reset system to default state.
        #touch /run/watchguard/dimension/wg_event_system_reset
        /opt/watchguard/dimension/bin/wg-system-reset.sh
        exit 0
    fi

    # Process system.json first
    _process_file "${WG_SYSDIR}/system.json"

    # Process the interfaces.json next
    _process_file "${WG_SYSDIR}/interfaces.json"

    # Process interface-*.json files next
    for f in "${WG_SYSDIR}"/interface-*.json ; do 
        _process_file "${f}"
    done

    # Process remaining files.
    for f in "${WG_SYSDIR}"/*.json ; do
        if _is_system_file "$f" || _is_interfaces_file "$f" \
                || _is_ntpd_file "$f" || _is_resolv_file "$f" \
                || _is_sshd_file "$f" \
                || _is_routes_file "$f" \
                || _is_interface_file "$f" ; then
            continue
        fi
        _process_file "${f}"
    done
fi

if [ ! -d "${WG_VARDIR}/data/tmp" ]; then
    _log_notice "Creating ${WG_VARDIR}/data/tmp"

    # Create temporary directory
    /bin/mkdir ${WG_VARDIR}/data/tmp
    chmod -R --reference=${WG_VARDIR} ${WG_VARDIR}/data/tmp
    chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/data/tmp
elif [ -d "${WG_VARDIR}/data/tmp" ]; then
    _log_notice "Cleaning ${WG_VARDIR}/data/tmp"

    # Make sure temporary directory is empty
    /bin/rm -rf ${WG_VARDIR}/data/tmp/*
fi

if [ -d "${WG_VARDIR}/data/session" ]; then
    # Make sure Web sessions directory is empty
    _log_notice "Cleaning ${WG_VARDIR}/data/session"
    rm -f ${WG_VARDIR}/data/session/*
fi

# Emit event to indicate system has been initialized.
#FIXME:
#dbus-send --system --type=method_call --dest=com.watchguard.system /com/watchguard/system/EventObject com.watchguard.system.EventHandler.SystemReady
