/*global active_backup_list, BackupLogObj, createDateAtMidnight,
  createDateFromStr, escapeHtml, is_page_locked, is_email_enabled, remote_backup_enabled,
  unescapeHtml, S_BACKUP_ADD_MSG, S_BACKUP_EDIT_MSG, S_BACKUP_EMPTY_TIME,
  S_BACKUP_ERROR_TIME, S_BACKUP_MESSAGE, S_BACKUP_NAME_DUPLICATE_MSG,
  S_BACKUP_NAME_MSG, S_BACKUP_RUN_AGAIN, S_BACKUP_TITLE
*/
var S_MT_B = {
    mode : '',
    complete_cb : null,
    modify_enabled : false,
    start_over : false,

    // setup the page
    init: function () {
        S_MT_B.initData();
        S_MT_B.initUI();
        S_MT_B.initEvents();
    },

    initData : function () {
        // data values are shown in html elements, so they need to be escaped before being displayed
        // we escape the data here and store them escaped at the client
        // data sent to the server must not be escaped
        // the javascript logic must consider whether a value is escaped or unescaped when working with it
        var i, len;

        for (i = 0, len = active_backup_list.length; i < len; i++) {
            active_backup_list[i].task_name = escapeHtml(active_backup_list[i].task_name);
            active_backup_list[i].description = escapeHtml(active_backup_list[i].description);
        }
    },

    // initialize UI elements
    initUI: function () {
        // disable the controls/buttons if remote backups are disabled
        if (!remote_backup_enabled) {
            WGRD.enableUIElements(false, ['#backup_name', '#backup_desc', '#start_date_val', '#end_date_val', '#backup_now']);
        }

        $('#backup_close').hide();

        $("#start_date span, #end_date span").bind('click', function (e) {
            if ($('#start_date_val').hasClass('disabled')) {
                e.stopImmediatePropagation();
            }
        });

        $('#start_date').datetimepicker({
            pickDate: true,
            pickTime: false
        });

        // we only support daily backups, where the day is a 24-hour period starting at 00:00:00 UTC.
        $('#start_date').data('datetimepicker').setLocalDate(createDateAtMidnight(0));

        $('#end_date').datetimepicker({
            pickDate: true,
            pickTime: false
        });
        $('#end_date').data('datetimepicker').setLocalDate(createDateAtMidnight(1));

        $('#email_notify_tooltip').tooltip();

        // detect if any fields are changed
        var flds = $("input", window.document);
        $(flds).on('change keyup', flds, function (eo) {
            $('#add_error_backup').hide();
        });
    },

    // add event handlers
    initEvents: function () {
        // detect if the settings changes when user edits the page
        $("#create_back_dlg input").change(S_MT_B.toggleModifyEnable);

        $('#backup_close').button().click(S_MT_B.closeTaskModal);
        $('#backup_now').button().click(S_MT_B.backupDb);
    },

    showTaskModal: function (params) {
        if (remote_backup_enabled) {
            if (S_MT_B.complete_cb === null) {
                S_MT_B.complete_cb = params.complete_cb;
            }
            WGRD.doModal('#create_back_dlg',
                         S_MT_B.initializeBackup,
                         null,
                         params);
        } else {
            // show a notification dialog telling the admin that system backup configuration is required
            WGRD.okMessageModal(S_BACKUP_MESSAGE, S_BACKUP_TITLE);
        }
    },

    closeTaskModal : function () {
        $('#create_back_dlg').modal('hide');
    },

    initializeBackup : function (params) {
        S_MT_B.mode = params.mode;
        if (params.mode === 'edit') {
            S_MT_B.editBackup(params);
        }
        if (is_email_enabled === 1) {
            $('#email_notify_chkbox').prop("disabled", false);
        } else {
            $('#email_notify_chkbox').prop("disabled", true);
        }
        return true;
    },

    toggleModifyEnable : function () {
        if (S_MT_B.modify_enabled && S_MT_B.mode === 'edit') {
            WGRD.enableUIElements(true, ['#backup_now']);
        }
    },

    toggleStatusAction : function (task_object) {
        //get task_status and run mode
        var task_status = task_object.status_value.substring(0, 3);
        var task_mode = task_object.run_mode;

        S_MT_B.start_over = true;
        // check the task status, to enable/disable run again button
        if (task_status === '101' || task_status === '102') {
            // change the text to run again
            $('#backup_now').text(S_BACKUP_RUN_AGAIN);
        } else if (task_status === '100') {
            // '1m': '1 month', '1w': '1 week', '1d': '1 day', '0d': 'one time'
            // The finished 'run one time' task could run again if any change to it. 
            if (task_mode === '0d') {
                S_MT_B.modify_enabled = true;
                $('#backup_now').attr('disabled', true);
            }
        } else if (task_status !== '103') {
            $('#backup_now').attr('disabled', true);
        }
    },

    toggleChangeViewMode : function () {
        WGRD.enableUIElements(false, ['#backup_name']);

        var enabled = true;
        // Disable or enable the other element ids in RO or RW mode.
        if (is_page_locked === 0) {
            enabled = false;
        }

        var elem_ids = ['#backup_desc', '#email_notify_chkbox'];
        WGRD.enableUIElements(enabled, elem_ids);
        if (enabled) {
            $('#backup_close').hide();
            $("#backup_now").show();
        } else {
            $('#backup_close').show();
            $("#backup_now").hide();
        }
    },

    editBackup : function (params) {
        var task_obj, task_name;
        var i, len;
        var active_obj, notify;

        // check the params
        task_obj = params.task;
        if (task_obj === undefined || task_obj === null) {
            return;
        }

        // handle the task status action
        S_MT_B.toggleStatusAction(task_obj);

        // handle the element ids
        S_MT_B.toggleChangeViewMode();

        task_name = task_obj.name;
        $('.wizard-subtitle').text(task_name);
        $('#backup_name').val(task_name);

        // fill decriptions, the start time and end time, notification checked or not
        for (i = 0, len = active_backup_list.length; i < len; i++) {
            active_obj = active_backup_list[i];
            if (active_obj.task_name === escapeHtml(task_name)) {
                // decriptions
                $('#backup_desc').val(unescapeHtml(active_obj.description));

                // start_time and end_time
                $('#start_date').data('datetimepicker').setDate(createDateFromStr(active_obj.data_begin_time));
                $('#end_date').data('datetimepicker').setDate(createDateFromStr(active_obj.data_end_time));

                // notify checkbox
                notify = active_obj.notify_admin;
                $('#email_notify_chkbox').prop('checked', notify);
                break;
            }
        }
    },

    // Add a new backup logs task
    backupDb : function () {
        // Verify the backup task name
        var backup_name = $('#backup_name').val();
        var backup_desc = $('#backup_desc').val();
        var backup_obj = null;
        var active_obj;
        var start_date, end_date;
        var notify_checked;
        var i, len;
        if (backup_name.trim().length === 0) {
            $('#backup_name').focus();
            $('#backup_name').select();

            $('#add_error_backup').show().text(S_BACKUP_NAME_MSG);
            return;
        }

        //load all of backup tasks
        for (i = 0, len = active_backup_list.length; i < len; i++) {
            active_obj = active_backup_list[i];
            if (active_obj.task_name.toLowerCase() === escapeHtml(backup_name.toLowerCase())) {
                //Check if backup name exists already.
                if (S_MT_B.mode === 'add') {
                    $('#add_error_backup').show().text(S_BACKUP_NAME_DUPLICATE_MSG);
                    return;
                }

                backup_obj = active_obj;
                break;
            }
        }

        // verify the start time is before the end time
        // note: start time and end time can be empty
        start_date = $('#start_date').data('datetimepicker').getLocalDate();
        end_date = $('#end_date').data('datetimepicker').getLocalDate();
        if (start_date === null || end_date === null) {
            $('#add_error_backup').show().text(S_BACKUP_EMPTY_TIME);
            return;
        }
        // only compare the date
        start_date.setHours(0, 0, 0, 0);
        end_date.setHours(0, 0, 0, 0);
        if (start_date.getTime() >= end_date.getTime()) {
            $('#add_error_backup').show().text(S_BACKUP_ERROR_TIME);
            return;
        }
        // get email notify checkbox
        notify_checked = $('#email_notify_chkbox').prop('checked');

        // create new object when the mode is 'add'.
        if (S_MT_B.mode === 'add') {
            backup_obj = new BackupLogObj();
        }

        $('#backup_now').attr('disabled', true);

        // update the current backup object, it will be saved to server.
        backup_obj.task_name = backup_name;
        backup_obj.description = backup_desc;
        backup_obj.run_interval = '0d';
        // the start date and end date will be saved as the string in UTC
        backup_obj.data_begin_time = start_date.format('yyyy-mm-dd HH:MM:ss');
        backup_obj.data_end_time = end_date.format('yyyy-mm-dd HH:MM:ss');
        backup_obj.notify_admin = notify_checked;

        // get the start over flag
        backup_obj.start_over = S_MT_B.start_over;

        // save the data to the server
        WGRD.putData(backup_obj).success(function (response) {
            S_MT_B.toggleCallBackAction();

            // we need to do some things after the save operations is successful
            if (response.status) {
                // unescaped data was sent to server, so we need to escape the local data.
                backup_obj.task_name = escapeHtml(backup_obj.task_name);
                backup_obj.description = escapeHtml(backup_obj.description);
                if (S_MT_B.mode === 'add') {
                    // append to the active backup task list
                    active_backup_list.push(backup_obj);
                    $('#error_container').text(S_BACKUP_ADD_MSG);
                } else {
                    $('#error_container').text(S_BACKUP_EDIT_MSG);
                }
            }
            $('#error_container')
                                .delay(2000)
                                .fadeOut();
        });

        S_MT_B.toggleBackupAction();
    },

    toggleBackupAction : function () {
        $('#create_back_dlg').modal('hide');
    },

    toggleCallBackAction : function () {
        // Call the callback function if it is set.
        if (S_MT_B.complete_cb !== null) {
            S_MT_B.complete_cb();
        }
    }

};
// Add the sourceURL for debug
//@ sourceURL=backup.js
$(document).ready(S_MT_B.init);
