#!/bin/sh
# wg_event.sh - Shell script to process WatchGuard system monitoring events

if [ $# -lt 3 ]; then
    echo 'Usage: wg_event.sh [config-change|task-request|ca-certs-update] [file] [op]'
    exit 0
fi

WG_EVENT_TYPE=${1}
WG_FILE=${2}
WG_OP=${3}

_log_error() {
    logger -s -p local0.err -t "wg_event" "ERROR: ${1}"
}

_log_notice() {
    logger -s -p local0.notice -t "wg_event" "NOTICE: ${1}"
}

_cmp_fname() {
    if [ ! -z "${1}"  -a ! -z "${2}" ] ; then
        fname="`basename ${1}`"
        if [ "$fname" = "${2}" ] ; then
            return 0
        fi
    fi
    return 1
}

_is_system_file(){
    _cmp_fname "${1}" "system.json"
}

. /etc/default/wg_system

WG_PYTHON_BIN=${WG_BINDIR}/python
WG_CONFIG_PYC=${WG_BINDIR}/wg_system_config.pyc
WG_TASKREQ_PYC=${WG_BINDIR}/wg_system_task.pyc

WG_CONFIG_DIR=${WG_ETCDIR}/system
WG_DEFAULTS_DIR=${WG_CONFIG_DIR}/defaults

UPDATE_CA_CERTIFICATES=/usr/sbin/update-ca-certificates

if [ ! -f "${WG_PYTHON_BIN}" -o ! -x "${WG_PYTHON_BIN}" ]; then
    _log_error "{WG_PYTHON_BIN} does not exist or is not an executable."
    exit 1
fi

if [ -z "${WG_FILE}" ] ; then
    _log_error "WG_FILE environment variable is not set."
    exit 1
fi

# Process wg_event notifications for .json files only.
if echo "${WG_FILE}" |  egrep -q '\.json$|\.crt$' ; then
    :
else
    exit 0
fi

case "$WG_EVENT_TYPE" in
"config-change")
    if [ "${WG_OP}" = "Deleted" ] && _is_system_file "${WG_FILE}" ; then
        # If the system.json file is deleted, we need to reset to
        # factory default, but we can't do it reliably within this task
        # instance since wg_systemd is waiting for it to complete. So
        # we emit an event to trigger another task to do the needful and
        # return.
        if [ -d "${WG_DEFAULTS_DIR}" ] ; then
            _log_notice "Emitting event to reset system configuration to default"
            dbus-send --system --type=method_call --dest=com.watchguard.system /com/watchguard/system/EventObject com.watchguard.system.EventHandler.SystemReset
            exit 0
        else
            _log_error "Directory ${WG_DEFAULTS_DIR} does not exist."
            exit 1
        fi
    fi

    if [ ! -f "${WG_CONFIG_PYC}" -o ! -s "${WG_CONFIG_PYC}" ]; then
        _log_error "File ${WG_CONFIG_PYC} does not exist or is empty."
        exit 1
    fi
    "${WG_PYTHON_BIN}" "${WG_CONFIG_PYC}" "${WG_FILE}"
    ;;

"task-request")
    if [ "${WG_OP}" = "Deleted" ]; then
        # Nothing to do when task-request files are deleted.
        exit 0
    fi
    if [ ! -f "${WG_TASKREQ_PYC}" -o ! -s "${WG_TASKREQ_PYC}" ]; then
        _log_error "File ${WG_TASKREQ_PYC} does not exist or is empty."
        exit 1
    fi
    "${WG_PYTHON_BIN}" "${WG_TASKREQ_PYC}" "${WG_FILE}"
    ;;

"ca-certs-update")
    if [ ! -x "${UPDATE_CA_CERTIFICATES}" ] ; then
        _log_error "File ${UPDATE_CA_CERTIFICATES} does not exist or is not executable."
        exit 1
    fi
    ${UPDATE_CA_CERTIFICATES} -f
    ;;

*)
    #Ignore unreqcognized event types.
    exit 0
    ;;
esac

